
#ifndef DX_SECURE_BOOT_H
#define DX_SECURE_BOOT_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "DX_VOS_BaseTypes.h"
#include "SEC_BOOT_RSA.h"
#include "CRYS_HASH.h"
#include "SEC_BOOT_error.h"

#ifdef __cplusplus
extern "C"
{
#endif



  /*
   *  Object name     : SecureBootMain.h
   *  State           :  %state%
   *  Creation date   :  Wed Feb 23 16:19:14 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version SecureBootMain.h#1:incl:1
   *  \author adams
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */


/************************ Defines ******************************/

#define VRL_MAGIC_NUMBER 0xE59FF010

/* The VRL primary table's layout is as follows:

   case 1: the VRL secondary table exists: 
   
            In this case the VRL table contains the following structures:
            1 - The header: VRL_PrimaryHeader_t
            2 - An array of the record indexes: VRL_RecordInfo_t
            3 - Secondary table info: VRL_SecondaryInfo_t
            4 - The signature: VRL_PrimarySign_t.
            
   case 2: the VRL secondary table does not exist:           

            In this case the VRL table contains the following structures:
            1 - The header: VRL_PrimaryHeader_t
            2 - An array of the record indexes: VRL_RecordInfo_t
            4 - The signature: VRL_PrimarySign_t. 
   
   The secondary VRL's layout is as follows:
   
            1 - The header: VRL_PrimaryHeader_t
            2 - An array of the record indexes: VRL_RecordInfo_t
            4 - The signature: VRL_PrimarySign_t. 
          
   The structures are defined below.
 */           

/* 1 . the VRL structure */
typedef struct 
{
   DxUint32_t MagicNumber;  
   DxUint32_t LenInWords;
   DxUint32_t SecondaryVrlPresent;
   DxUint32_t SecondaryVrlAddr;
   DxUint32_t MemCopySizeInBytes;
   DxUint32_t NumOfRecords;

}VRL_Header_t;
   
/* 2. the record structure */
typedef struct
{
   DxUint32_t Addr;
   DxUint32_t LenInWords;
   DxUint32_t CheckSwOnWakeUpMode;
   HASH_Result_t SwHashResult;  

}VRL_RecordInfo_t; 

/* 3. secondary info on the primary structure */
typedef struct
{
   DxUint32_t N[RSA_MOD_SIZE_IN_BITS/32];

}VRL_SecondaryInfo_t;

/* 4. the signature */
typedef struct
{
   DxUint32_t H[RSA_MOD_SIZE_IN_BITS/32];
   DxUint32_t sig[RSA_MOD_SIZE_IN_BITS/32];
   
}VRL_Sign_t;         

#define DX_NO_VERIFICATION_REQUIRED 0xFFFF

/**
 * @brief The DX_SecureBoot function executes the secured boot process.
 *
 * The function verifies the integrity of the data listed in the VRL,
 * as descirbed in the Secure Boot Specification document.
 * 
 * @param[in] VrlPrimaryAddr - the Address of the VRL on the flash
 * @param[in] GetFlashData - Pointer to function for reading flash
 * @param[in] N_ptr - the pointer to the pub key mod. 
 * @param[in] Index - The index determining to what key stored in the OTP the pub key N should be verified
 *            the valid values are 0-4 , If the value is set to DX_NO_VERIFICATION_REQUIRED no verification is executed. 
 * @param[in] WorkRam_ptr - Pointer to scratchpad RAM
 *
 * @return DX_OK if the verification succeeded
 *         DX_INVALID_PRIMARY_VRL_MAGIC_NUM_VAL if VrlPrimaryAddr does not point to a valid VRL table
 *         DX_INVALID_SECONDARY_VRL_MAGIC_NUM_VAL the secondary VRL table (pointed to by the primary VRL) is invalid
 *         DX_VRL_PRIMARY_FAILED or 
 *         DX_VRL_SECONDARY_FAILED if the primary or secondary verification failed, respectively
 *         DX_RECORD_INTEGRITY_FAILURE(T,N) if record N of table T failed verification
 *
 */                               
DX_Error_t DX_SecureBoot ( DxUint32_t VrlPrimaryAddr,
                           void (*GetFlashData)(DxUint32_t *,DxUint32_t,DxUint32_t),
                           DxUint32_t *N_ptr,
                           DxUint32_t Index, 
                           DxUint32_t *WorkRam_ptr );

#ifdef __cplusplus
}
#endif

#endif

